<?php
/**
 * Gerador de arquivos para oferta
 * Gera ZIP com todos os arquivos necessários
 */
class OfferGenerator {
    private $db;
    private $offer;
    private $gateway;
    
    public function __construct($db) {
        $this->db = $db;
    }
    
    public function generate($offerId) {
        // Busca oferta
        $this->offer = $this->db->fetchOne("SELECT * FROM offers WHERE id = ?", [$offerId]);
        if (!$this->offer) throw new Exception('Oferta não encontrada');
        
        // Busca gateway
        if ($this->offer['gateway_id']) {
            $this->gateway = $this->db->fetchOne("SELECT * FROM gateways WHERE id = ?", [$this->offer['gateway_id']]);
        } else {
            $this->gateway = $this->db->fetchOne("SELECT * FROM gateways WHERE is_default = 1");
        }
        
        // Busca domínios
        $domains = $this->db->fetchAll("SELECT domain FROM domains WHERE offer_id = ?", [$offerId]);
        
        // Cria diretório temporário
        $tmpDir = sys_get_temp_dir() . '/offer_' . $offerId . '_' . time();
        mkdir($tmpDir, 0755, true);
        mkdir($tmpDir . '/api', 0755);
        mkdir($tmpDir . '/assets', 0755);
        mkdir($tmpDir . '/data', 0755);
        mkdir($tmpDir . '/logs', 0755);
        
        // Gera arquivos
        file_put_contents($tmpDir . '/config.php', $this->generateConfig());
        file_put_contents($tmpDir . '/index.php', $this->generateIndex());
        file_put_contents($tmpDir . '/CloakerPro.php', $this->generateCloaker());
        file_put_contents($tmpDir . '/api/pix.php', $this->generatePixApi());
        file_put_contents($tmpDir . '/api/webhook.php', $this->generateWebhook());
        file_put_contents($tmpDir . '/api/status.php', $this->generateStatus());
        file_put_contents($tmpDir . '/assets/fingerprint.js', $this->generateFingerprint());
        file_put_contents($tmpDir . '/assets/tracking.js', $this->generateTracking());
        file_put_contents($tmpDir . '/.htaccess', $this->generateHtaccess());
        file_put_contents($tmpDir . '/data/.gitkeep', '');
        file_put_contents($tmpDir . '/logs/.gitkeep', '');
        
        // Cria ZIP
        $zipPath = sys_get_temp_dir() . '/' . $this->offer['slug'] . '.zip';
        $zip = new ZipArchive();
        $zip->open($zipPath, ZipArchive::CREATE | ZipArchive::OVERWRITE);
        
        $this->addDirToZip($zip, $tmpDir, '');
        $zip->close();
        
        // Limpa temporário
        $this->deleteDir($tmpDir);
        
        // Atualiza timestamp
        $this->db->update('offers', ['files_generated_at' => date('Y-m-d H:i:s')], 'id = ?', [$offerId]);
        
        return $zipPath;
    }
    
    private function generateConfig() {
        $o = $this->offer;
        $g = $this->gateway ?: [];
        
        $apiKey = $o['gateway_custom_key'] ?: ($g['api_key'] ?? '');
        $companyId = $o['gateway_custom_company'] ?: ($g['company_id'] ?? '');
        
        return '<?php
/**
 * =============================================
 * CONFIGURAÇÃO: ' . $o['name'] . '
 * Gerado em: ' . date('Y-m-d H:i:s') . '
 * =============================================
 */

// Identificação
define("OFFER_ID", ' . $o['id'] . ');
define("OFFER_SLUG", "' . $o['slug'] . '");
define("OFFER_NAME", "' . addslashes($o['name']) . '");

// URLs
define("PAGE_REAL", "' . $o['real_page_url'] . '");
define("PAGE_WHITE", "' . $o['white_page_url'] . '");

// Preço
define("PRICE", ' . floatval($o['price_per_number']) . ');

// ============ CLOAKER ============
define("CLOAKER_ENABLED", ' . ($o['cloaker_enabled'] ? 'true' : 'false') . ');
define("TRAFFIC_TOKEN", "' . $o['traffic_token'] . '");
define("TOKEN_PARAM", "' . $o['token_param'] . '");
define("REQUIRE_TOKEN", ' . ($o['require_token'] ? 'true' : 'false') . ');
define("MIN_TIME_ON_PAGE", ' . intval($o['min_time_on_page']) . ');

// Proteção
define("BLOCK_VPN", ' . ($o['block_vpn'] ? 'true' : 'false') . ');
define("BLOCK_DATACENTER", ' . ($o['block_datacenter'] ? 'true' : 'false') . ');
define("BLOCK_TOR", ' . ($o['block_tor'] ? 'true' : 'false') . ');
define("USE_FINGERPRINT", ' . ($o['use_fingerprint'] ? 'true' : 'false') . ');
define("USE_HONEYPOT", ' . ($o['use_honeypot'] ? 'true' : 'false') . ');

// Geo
define("ALLOWED_COUNTRIES", \'' . ($o['allowed_countries'] ?: '["BR"]') . '\');
define("ALLOW_MOBILE", ' . ($o['allow_mobile'] ? 'true' : 'false') . ');
define("ALLOW_DESKTOP", ' . ($o['allow_desktop'] ? 'true' : 'false') . ');
define("ALLOW_TABLET", ' . ($o['allow_tablet'] ? 'true' : 'false') . ');

// ============ GATEWAY ============
define("GATEWAY_NAME", "' . ($g['name'] ?? 'Default') . '");
define("GATEWAY_URL", "' . ($g['api_url'] ?? '') . '");
define("GATEWAY_KEY", "' . $apiKey . '");
define("GATEWAY_COMPANY", "' . $companyId . '");
define("GATEWAY_AUTH_TYPE", "' . ($g['auth_type'] ?? 'apikey') . '");
define("GATEWAY_AUTH_HEADER", "' . ($g['auth_header'] ?? 'x-api-key') . '");
define("GATEWAY_AUTH_PREFIX", "' . ($g['auth_prefix'] ?? '') . '");
define("GATEWAY_ENDPOINT_CREATE", "' . ($g['endpoint_create'] ?? '/api/transactions') . '");
define("GATEWAY_STATUS_FIELD", "' . ($g['status_field'] ?? 'status') . '");
define("GATEWAY_QRCODE_FIELD", "' . ($g['qrcode_field'] ?? 'pix.qrcode') . '");
define("GATEWAY_COPYPASTE_FIELD", "' . ($g['copypaste_field'] ?? 'pix.copiaECola') . '");
define("GATEWAY_TXID_FIELD", "' . ($g['txid_field'] ?? 'id') . '");
define("GATEWAY_VALOR_REAIS", ' . (($g['valor_em_reais'] ?? 0) ? 'true' : 'false') . ');

// ============ FACEBOOK PIXEL ============
define("FB_PIXEL_ID", "' . $o['fb_pixel_id'] . '");
define("FB_ACCESS_TOKEN", "' . $o['fb_access_token'] . '");
define("FB_TEST_CODE", "' . $o['fb_test_event_code'] . '");
define("FB_CAPI_ENABLED", ' . ($o['fb_capi_enabled'] ? 'true' : 'false') . ');

// ============ UTMIFY ============
define("UTMIFY_ENABLED", ' . ($o['utmify_enabled'] ? 'true' : 'false') . ');
define("UTMIFY_TOKEN", "' . $o['utmify_token'] . '");
define("UTMIFY_PROJECT", "' . $o['utmify_project_id'] . '");

// ============ GOOGLE ADS ============
define("GADS_ENABLED", ' . ($o['gads_enabled'] ? 'true' : 'false') . ');
define("GADS_ID", "' . $o['gads_conversion_id'] . '");
define("GADS_LABEL", "' . $o['gads_conversion_label'] . '");

// ============ TIKTOK ============
define("TIKTOK_PIXEL", "' . $o['tiktok_pixel_id'] . '");
define("TIKTOK_TOKEN", "' . $o['tiktok_access_token'] . '");

// ============ PROTEÇÃO CHECKOUT ============
define("PROTECTION_SECRET", "RIFA_PROTECT_2025");

// Diretórios
define("DATA_DIR", __DIR__ . "/data/");
define("LOGS_DIR", __DIR__ . "/logs/");

date_default_timezone_set("America/Sao_Paulo");
';
    }
    
    private function generateIndex() {
        return '<?php
require_once __DIR__ . "/config.php";
require_once __DIR__ . "/CloakerPro.php";

$cloaker = new CloakerPro();
$result = $cloaker->process();

if ($result["passed"]) {
    // Gera token de proteção para o checkout
    $protectionToken = md5(PROTECTION_SECRET . date("Y-m-d"));
    
    // Monta URL com token de proteção
    $params = $_GET;
    $params["_t"] = $protectionToken;
    
    $url = PAGE_REAL;
    $url .= (strpos($url, "?") ? "&" : "?") . http_build_query($params);
    
    header("Location: " . $url);
} else {
    header("Location: " . PAGE_WHITE);
}
exit;
';
    }
    
    private function generateCloaker() {
        return '<?php
class CloakerPro {
    private $score = 100;
    private $reasons = [];
    
    public function process() {
        if (!CLOAKER_ENABLED) return ["passed" => true, "score" => 100];
        
        $ip = $this->getIP();
        $ua = $_SERVER["HTTP_USER_AGENT"] ?? "";
        
        // Token
        if (REQUIRE_TOKEN) {
            $token = $_GET[TOKEN_PARAM] ?? "";
            if ($token !== TRAFFIC_TOKEN) {
                $this->score -= 50;
                $this->reasons[] = "invalid_token";
            }
        }
        
        // Bot
        if ($this->isBot($ua)) {
            $this->score -= 80;
            $this->reasons[] = "bot";
        }
        
        // Datacenter
        if (BLOCK_DATACENTER && $this->isDatacenter($ip)) {
            $this->score -= 60;
            $this->reasons[] = "datacenter";
        }
        
        // Geo
        $country = $this->getCountry($ip);
        $allowed = json_decode(ALLOWED_COUNTRIES, true) ?: ["BR"];
        if (!in_array($country, $allowed)) {
            $this->score -= 70;
            $this->reasons[] = "geo:" . $country;
        }
        
        // Device
        $device = $this->getDevice($ua);
        if ($device === "mobile" && !ALLOW_MOBILE) { $this->score -= 40; $this->reasons[] = "mobile_blocked"; }
        if ($device === "desktop" && !ALLOW_DESKTOP) { $this->score -= 40; $this->reasons[] = "desktop_blocked"; }
        if ($device === "tablet" && !ALLOW_TABLET) { $this->score -= 40; $this->reasons[] = "tablet_blocked"; }
        
        // Log
        $this->log($ip, $ua, $country, $device);
        
        $passed = $this->score >= 50;
        return ["passed" => $passed, "score" => $this->score, "reasons" => $this->reasons];
    }
    
    private function getIP() {
        foreach (["HTTP_CF_CONNECTING_IP", "HTTP_X_FORWARDED_FOR", "HTTP_X_REAL_IP", "REMOTE_ADDR"] as $h) {
            if (!empty($_SERVER[$h])) return explode(",", $_SERVER[$h])[0];
        }
        return "0.0.0.0";
    }
    
    private function isBot($ua) {
        $bots = ["bot", "crawler", "spider", "facebook", "google", "bing", "curl", "wget", "python", "headless", "phantom"];
        $ua = strtolower($ua);
        foreach ($bots as $b) if (strpos($ua, $b) !== false) return true;
        return false;
    }
    
    private function isDatacenter($ip) {
        $ranges = ["34.", "35.", "52.", "54.", "13.", "18.", "157.240.", "66.220.", "69.171.", "173.252."];
        foreach ($ranges as $r) if (strpos($ip, $r) === 0) return true;
        return false;
    }
    
    private function getCountry($ip) {
        $cache = DATA_DIR . "geo_cache.json";
        $data = file_exists($cache) ? json_decode(file_get_contents($cache), true) : [];
        if (isset($data[$ip])) return $data[$ip];
        
        $res = @file_get_contents("http://ip-api.com/json/{$ip}?fields=countryCode");
        $json = json_decode($res, true);
        $country = $json["countryCode"] ?? "XX";
        
        $data[$ip] = $country;
        file_put_contents($cache, json_encode($data));
        return $country;
    }
    
    private function getDevice($ua) {
        $ua = strtolower($ua);
        if (preg_match("/tablet|ipad|playbook|silk/", $ua)) return "tablet";
        if (preg_match("/mobile|android|iphone|ipod|opera mini|opera mobi/", $ua)) return "mobile";
        return "desktop";
    }
    
    private function log($ip, $ua, $country, $device) {
        $log = date("Y-m-d H:i:s") . "|{$ip}|{$country}|{$device}|" . ($this->score >= 50 ? "PASS" : "BLOCK") . "|" . implode(",", $this->reasons) . "\n";
        @file_put_contents(LOGS_DIR . date("Y-m-d") . ".log", $log, FILE_APPEND);
    }
}
';
    }
    
    private function generatePixApi() {
        return '<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");

if ($_SERVER["REQUEST_METHOD"] === "OPTIONS") exit;

require_once __DIR__ . "/../config.php";

$input = json_decode(file_get_contents("php://input"), true);

$name = trim($input["name"] ?? "");
$email = trim($input["email"] ?? "");
$cpf = preg_replace("/[^0-9]/", "", $input["cpf"] ?? "");
$phone = preg_replace("/[^0-9]/", "", $input["phone"] ?? "");
$amount = floatval($input["amount"] ?? 0);
$qty = intval($input["quantity"] ?? 1);

if (!$name || !$email || !$cpf || $amount <= 0) {
    die(json_encode(["success" => false, "error" => "Dados incompletos"]));
}

$txRef = "TX" . OFFER_ID . "_" . time() . rand(100, 999);
$amountValue = GATEWAY_VALOR_REAIS ? $amount : intval($amount * 100);

$payload = [
    "amount" => $amountValue,
    "paymentMethod" => "PIX",
    "customer" => ["name" => $name, "email" => $email, "document" => $cpf, "phone" => $phone],
    "externalReference" => $txRef,
    "callbackUrl" => "https://" . $_SERVER["HTTP_HOST"] . dirname($_SERVER["REQUEST_URI"]) . "/webhook.php"
];

$headers = ["Content-Type: application/json"];

if (GATEWAY_AUTH_TYPE === "basic") {
    $auth = base64_encode(GATEWAY_COMPANY . ":" . GATEWAY_KEY);
    $headers[] = GATEWAY_AUTH_HEADER . ": Basic " . $auth;
} elseif (GATEWAY_AUTH_TYPE === "bearer") {
    $headers[] = GATEWAY_AUTH_HEADER . ": Bearer " . GATEWAY_KEY;
} else {
    $prefix = GATEWAY_AUTH_PREFIX ? GATEWAY_AUTH_PREFIX . " " : "";
    $headers[] = GATEWAY_AUTH_HEADER . ": " . $prefix . GATEWAY_KEY;
}

$ch = curl_init(GATEWAY_URL . GATEWAY_ENDPOINT_CREATE);
curl_setopt_array($ch, [
    CURLOPT_POST => true,
    CURLOPT_POSTFIELDS => json_encode($payload),
    CURLOPT_HTTPHEADER => $headers,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_TIMEOUT => 30
]);
$response = curl_exec($ch);
curl_close($ch);

$data = json_decode($response, true);

$qrcode = extractPath($data, GATEWAY_QRCODE_FIELD);
$copypaste = extractPath($data, GATEWAY_COPYPASTE_FIELD);
$txid = extractPath($data, GATEWAY_TXID_FIELD);

if ($qrcode || $copypaste) {
    $tx = [
        "tx_ref" => $txRef,
        "gateway_tx_id" => $txid,
        "amount" => $amountValue,
        "customer" => compact("name", "email", "cpf", "phone"),
        "status" => "pending",
        "created_at" => date("Y-m-d H:i:s"),
        "utm" => [
            "source" => $input["utm_source"] ?? "",
            "medium" => $input["utm_medium"] ?? "",
            "campaign" => $input["utm_campaign"] ?? ""
        ]
    ];
    file_put_contents(DATA_DIR . "tx_" . $txRef . ".json", json_encode($tx));
    
    echo json_encode(["success" => true, "tx_ref" => $txRef, "qrcode" => $qrcode, "copypaste" => $copypaste, "txid" => $txid]);
} else {
    echo json_encode(["success" => false, "error" => "Erro gateway", "debug" => $data]);
}

function extractPath($data, $path) {
    if (!$path || !$data) return null;
    foreach (explode(".", $path) as $p) {
        if (preg_match("/^(.+)\[(\d+)\]$/", $p, $m)) $data = $data[$m[1]][$m[2]] ?? null;
        else $data = $data[$p] ?? null;
        if ($data === null) return null;
    }
    return $data;
}
';
    }
    
    private function generateWebhook() {
        return '<?php
require_once __DIR__ . "/../config.php";

$payload = file_get_contents("php://input");
$data = json_decode($payload, true);

file_put_contents(LOGS_DIR . "webhook_" . date("Y-m-d") . ".log", date("H:i:s") . " | " . $payload . "\n", FILE_APPEND);

$status = extractPath($data, GATEWAY_STATUS_FIELD);
$txRef = $data["externalReference"] ?? extractPath($data, GATEWAY_TXID_FIELD) ?? "";

if ($txRef) {
    $files = glob(DATA_DIR . "tx_*.json");
    foreach ($files as $f) {
        $tx = json_decode(file_get_contents($f), true);
        if ($tx["tx_ref"] === $txRef || $tx["gateway_tx_id"] === $txRef) {
            $tx["status"] = mapStatus($status);
            $tx["status_raw"] = $status;
            $tx["webhook_at"] = date("Y-m-d H:i:s");
            
            if (isPaid($status)) {
                $tx["paid_at"] = date("Y-m-d H:i:s");
                // Dispara eventos se configurado
                if (FB_CAPI_ENABLED && FB_ACCESS_TOKEN) sendFBEvent($tx);
                if (UTMIFY_ENABLED && UTMIFY_TOKEN) sendUtmify($tx);
            }
            
            file_put_contents($f, json_encode($tx));
            break;
        }
    }
}

echo json_encode(["received" => true]);

function extractPath($data, $path) {
    if (!$path || !$data) return null;
    foreach (explode(".", $path) as $p) {
        $data = $data[$p] ?? null;
        if ($data === null) return null;
    }
    return $data;
}

function mapStatus($s) {
    $map = ["PAID"=>"paid","paid"=>"paid","APPROVED"=>"paid","approved"=>"paid","COMPLETED"=>"paid","PENDING"=>"pending","pending"=>"pending","EXPIRED"=>"expired","CANCELLED"=>"cancelled","REFUNDED"=>"refunded"];
    return $map[strtoupper($s)] ?? "unknown";
}

function isPaid($s) {
    return in_array(strtoupper($s), ["PAID","APPROVED","COMPLETED","CONFIRMED","AUTHORIZED"]);
}

function sendFBEvent($tx) {
    $url = "https://graph.facebook.com/v18.0/" . FB_PIXEL_ID . "/events?access_token=" . FB_ACCESS_TOKEN;
    $event = [
        "data" => [[
            "event_name" => "Purchase",
            "event_time" => time(),
            "user_data" => ["em" => hash("sha256", $tx["customer"]["email"]), "ph" => hash("sha256", $tx["customer"]["phone"])],
            "custom_data" => ["currency" => "BRL", "value" => $tx["amount"] / 100]
        ]]
    ];
    if (FB_TEST_CODE) $event["test_event_code"] = FB_TEST_CODE;
    
    $ch = curl_init($url);
    curl_setopt_array($ch, [CURLOPT_POST => true, CURLOPT_POSTFIELDS => json_encode($event), CURLOPT_HTTPHEADER => ["Content-Type: application/json"], CURLOPT_RETURNTRANSFER => true]);
    curl_exec($ch);
    curl_close($ch);
}

function sendUtmify($tx) {
    // Implementar UTMify
}
';
    }
    
    private function generateStatus() {
        return '<?php
header("Content-Type: application/json");
require_once __DIR__ . "/../config.php";

$txRef = $_GET["tx_ref"] ?? "";
if (!$txRef) die(json_encode(["success" => false]));

$file = DATA_DIR . "tx_" . $txRef . ".json";
if (!file_exists($file)) die(json_encode(["success" => false, "error" => "Not found"]));

$tx = json_decode(file_get_contents($file), true);
echo json_encode(["success" => true, "status" => $tx["status"] ?? "pending", "paid_at" => $tx["paid_at"] ?? null]);
';
    }
    
    private function generateFingerprint() {
        return '(function(){window.CloakerFP={collect:function(){return btoa(JSON.stringify({s:screen.width+"x"+screen.height,t:Intl.DateTimeFormat().resolvedOptions().timeZone,l:navigator.language,p:navigator.platform}))}};})();';
    }
    
    private function generateTracking() {
        $o = $this->offer;
        $js = '';
        
        // Facebook Pixel
        if ($o['fb_pixel_id']) {
            $js .= "!function(f,b,e,v,n,t,s){if(f.fbq)return;n=f.fbq=function(){n.callMethod?n.callMethod.apply(n,arguments):n.queue.push(arguments)};if(!f._fbq)f._fbq=n;n.push=n;n.loaded=!0;n.version='2.0';n.queue=[];t=b.createElement(e);t.async=!0;t.src=v;s=b.getElementsByTagName(e)[0];s.parentNode.insertBefore(t,s)}(window,document,'script','https://connect.facebook.net/en_US/fbevents.js');fbq('init','" . $o['fb_pixel_id'] . "');fbq('track','PageView');\n";
        }
        
        // Google Ads
        if ($o['gads_enabled'] && $o['gads_conversion_id']) {
            $js .= "window.dataLayer=window.dataLayer||[];function gtag(){dataLayer.push(arguments);}gtag('js',new Date());gtag('config','" . $o['gads_conversion_id'] . "');\n";
        }
        
        // TikTok
        if ($o['tiktok_pixel_id']) {
            $js .= "!function(w,d,t){w.TiktokAnalyticsObject=t;var ttq=w[t]=w[t]||[];ttq.methods=['page','track','identify','instances','debug','on','off','once','ready','alias','group','enableCookie','disableCookie'];ttq.setAndDefer=function(t,e){t[e]=function(){t.push([e].concat(Array.prototype.slice.call(arguments,0)))}};for(var i=0;i<ttq.methods.length;i++)ttq.setAndDefer(ttq,ttq.methods[i]);ttq.instance=function(t){for(var e=ttq._i[t]||[],n=0;n<ttq.methods.length;n++)ttq.setAndDefer(e,ttq.methods[n]);return e};ttq.load=function(e,n){var i='https://analytics.tiktok.com/i18n/pixel/events.js';ttq._i=ttq._i||{};ttq._i[e]=[];ttq._i[e]._u=i;ttq._t=ttq._t||{};ttq._t[e]=+new Date;ttq._o=ttq._o||{};ttq._o[e]=n||{};var o=document.createElement('script');o.type='text/javascript';o.async=!0;o.src=i+'?sdkid='+e+'&lib='+t;var a=document.getElementsByTagName('script')[0];a.parentNode.insertBefore(o,a)};ttq.load('" . $o['tiktok_pixel_id'] . "');ttq.page();}(window,document,'ttq');\n";
        }
        
        return $js ?: '// No tracking configured';
    }
    
    private function generateHtaccess() {
        return 'RewriteEngine On
<FilesMatch "\.(json|log)$">
    Require all denied
</FilesMatch>
<IfModule mod_headers.c>
    Header set X-Content-Type-Options "nosniff"
    Header set X-Frame-Options "SAMEORIGIN"
</IfModule>
';
    }
    
    private function addDirToZip($zip, $dir, $base) {
        $files = scandir($dir);
        foreach ($files as $f) {
            if ($f === '.' || $f === '..') continue;
            $path = $dir . '/' . $f;
            $zipPath = $base ? $base . '/' . $f : $f;
            if (is_dir($path)) {
                $zip->addEmptyDir($zipPath);
                $this->addDirToZip($zip, $path, $zipPath);
            } else {
                $zip->addFile($path, $zipPath);
            }
        }
    }
    
    private function deleteDir($dir) {
        $files = scandir($dir);
        foreach ($files as $f) {
            if ($f === '.' || $f === '..') continue;
            $path = $dir . '/' . $f;
            is_dir($path) ? $this->deleteDir($path) : unlink($path);
        }
        rmdir($dir);
    }
}
